<?php
/**
 * 2007-2018 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 *  @author    PrestaShop SA <contact@prestashop.com>
 *  @copyright  2007-2022 PrestaShop SA
 *  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

class OrangeMoneyCmrPayment extends ObjectModel
{
    public $id_payment;
    public $ref_order;
    public $customer_name;
    public $customer_email;
    public $customer_tel;
    public $request_status;
    public $payment_amount;
    public $payment_status;
    public $message;
    public $pay_token;
    public $payment_url;
    public $notif_token;
    public $transaction_id;
    public $date_add;

    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'orangemoneycmr_payments',
        'primary' => 'id_payment',
        'multilang' => false,
        'fields' => array(
            'id_payment' => array('type' => self::TYPE_INT),
            'ref_order' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'customer_name' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'customer_email' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'customer_tel' => array('type' => self::TYPE_INT),
            'request_status' => array('type' => self::TYPE_INT),
            'payment_amount' => array('type' => self::TYPE_FLOAT),
            'payment_status' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'message' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'pay_token' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'payment_url' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'notif_token' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'transaction_id' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDateFormat'),
        )
    );


    /**
     * Send payment request
     */
    public static function sendPaymentResquest($order, $module, $customer_details)
    {

        $module->writeInlogFile($module->l('Initiating payment request ...'), 'INFO');
        $module->writeInlogFile($module->l('Setting payment request header ...'), 'INFO');
        $header = OrangeMoneyCmrPayment::setRequestHeader('payment_or_status_request', $module);
        if (!$header){
            $module->writeInlogFile($module->l('Setting payment request header failed'), 'ERROR');
            return false;
        }

        $module->writeInlogFile($module->l('Setting payment request body ...'), 'INFO');
        $data_string = OrangeMoneyCmrPayment::setRequestBody('payment_request', $module, $order);
        if (!$data_string){
            $module->writeInlogFile($module->l('Setting payment request body failed'), 'ERROR');
            return false;
        }

        $module->writeInlogFile($module->l('Sending payment request ...'), 'INFO');
        $results = OrangeMoneyCmrPayment::sendRequest($module, $header, $data_string, $module->paymentUrl);
        if (!$results) {
            return false;
        }

        switch ($results['message']) {
            case 'OK':
                $module->writeInlogFile($module->l('Payment request successfull !!!'), 'INFO');
                $module->writeInlogFile($module->l('Saving payment ...'), 'INFO');
                OrangeMoneyCmrPayment::savePayment($order, $results, $customer_details);
                $module->writeInlogFile($module->l('Payment saved !!!'), 'INFO');
                return $results['payment_url'];

            case 'Expired credentials':
                $module->writeInlogFile($module->l('Payment request failed with the following message: Expired credentials'), 'ERROR');
                $module->writeInlogFile($module->l('Genereting new access token: '), 'INFO');
                $token = OrangeMoneyCmrPayment::generateAccessToken($module);
                if ($token) {
                    $module->accessToken = $token;
                    OrangeMoneyCmrPayment::sendPaymentResquest($order, $module, $customer_details);
                } else {
                    return false;
                }

            default:
                return false;
        }
    }


    /**
     * Generates access token
     */
    public static function generateAccessToken($module)
    {
        $module->writeInlogFile($module->l('Initiating Access Token creation request ...'), 'INFO');
        $header = OrangeMoneyCmrPayment::setRequestHeader('access_token_request', $module);
        if (!$header){
            $module->writeInlogFile($module->l('Setting Access Token request header failed'), 'ERROR');
            return false;
        }

        $module->writeInlogFile($module->l('Setting Access Token request body ...'), 'INFO');
        $data_string = OrangeMoneyCmrPayment::setRequestBody('access_token_request', $module);
        if (!$data_string){
            $module->writeInlogFile($module->l('Setting Access Token request body failed'), 'ERROR');
            return false;
        }
       
        $module->writeInlogFile($module->l('Sending Access Token request ...'), 'INFO');
        $results = OrangeMoneyCmrPayment::sendRequest($module, $header, $data_string, $module->tokenUrl);
        if (!$results) {
            return false;
        }
        
        if ($results['access_token']) {
            $module->writeInlogFile($module->l('Token creation successfull !!! New Token = ').$results['access_token'], 'INFO');
            $module->writeInlogFile($module->l('Updating access token value in the configuration table ...'), 'INFO');
            Configuration::updateValue('AB_ACCESS_TOKEN', $results['access_token']);
            $module->writeInlogFile($module->l('Access token value updated !!!'), 'INFO');
            return $results['access_token'];

        } else {
            return false;
        }
    }
    
    /**
     * Save payment 
     */
    public static function savePayment($order, $results, $customer_details)
    {
        $payment = new OrangeMoneyCmrPayment();
        $payment->ref_order = $order->reference;
        $payment->customer_name = $customer_details['customer_name'];
        $payment->customer_email = $customer_details['customer_email'];
        $payment->customer_tel = $customer_details['customer_tel'];
        $payment->request_status = (int)$results['status'];
        $payment->payment_status = 'INITIATED';
        $payment->payment_amount = (float)$order->total_paid;
        $payment->message = $results['message'];
        $payment->pay_token = $results['pay_token'];
        $payment->payment_url = $results['payment_url'];
        $payment->notif_token = $results['notif_token'];
        $payment->save();
    }
    
    /**
     * Update payment
     */
    public static function updatePayment($payment_details, $paymentStatus, $transactionID)
    {
        $payment_details->payment_status = $paymentStatus;
        $payment_details->transaction_id = $transactionID;
        $payment_details->update();
    }


    /**
     * Gets payment details for a given Order
     */
    public static function getPaymentDetails($order_reference)
    {
        $q = 'SELECT id_payment FROM '._DB_PREFIX_.'orangemoneycmr_payments WHERE ref_order='."'$order_reference'";
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($q);
        $id = 0;

        if ($result) {
            $id = $result[0]['id_payment'];
        }
        if ($id != 0) {
            return new OrangeMoneyCmrPayment($id);
        }

        return false;
    }

    /**
     * Sets the request header
     */
    public static function setRequestHeader($operation, $module){

        $header=array();

        switch ($operation) {
            case 'payment_or_status_request':
                if (empty($module->accessToken)) {
                    $module->writeInlogFile($module->l('Setting payment request header failed because no access token was provided '), 'ERROR');
                    return false;
                }
                $header[]="Content-Type:application/json";
                $header[]="Accept:application/json";
                $header[]="Authorization:Bearer ".$module->accessToken;
                return $header;
            
            case 'access_token_request':
                if (empty($module->authHeader)) {
                    $module->writeInlogFile($module->l('Setting access token request header failed because no auth header was provided '), 'ERROR');
                    return false;
                }
                $header[]="Content-Type: application/x-www-form-urlencoded";
                $header[]="Authorization:".$module->authHeader;
                return $header;

            default:
                return false;
        }
    }

    /**
     * Sets request body
     */
    public static function setRequestBody($operation, $module, $order = null, $payment_details = null){

        switch ($operation) {
            case 'payment_request':
                $data = array();
                $data['merchant_key'] = $module->merchantKey;
                $data['currency'] = $module->currency;
                $data['order_id'] = $order->reference;
                $data['amount'] = (float)$order->total_paid;
                $data['return_url'] = $module->base_link.'?fc=module&module=orangemoneycmr&controller=finalisation&key=r'.$order->id;
                $data['cancel_url'] = $module->base_link.'?fc=module&module=orangemoneycmr&controller=finalisation&key=c'.$order->id;
                $data['notif_url'] = $module->base_link.'?fc=module&module=orangemoneycmr&controller=finalisation&key=n'.$order->id;
                $data['lang'] = $module->language;
                $data['reference'] = $module->merchantName;
                return json_encode($data);

            case 'status_request':
                $data = array();
                $data['order_id'] = $payment_details->ref_order;
                $data['amount'] = $payment_details->payment_amount;
                $data['pay_token'] = $payment_details->pay_token;
                return json_encode($data);

            case 'access_token_request':
                $data = "grant_type=client_credentials";
                return $data;

            default:
                return false;
        }
    }

    /**
     * Send cURL request
     */
    public static function sendRequest($module, $header, $data_string, $url){

        $module->writeInlogFile($module->l('Request header ').json_encode($header), 'INFO');
        $module->writeInlogFile($module->l('Request body ').json_encode($data_string), 'INFO');
        $module->writeInlogFile($module->l('Request URL ').$url, 'INFO');

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $header);

        $result_response = curl_exec($ch);

        $results = array();
        
        if (curl_errno($ch) > 0) {
            $module->writeInlogFile($module->l('There was a cURL error: '). curl_error($ch), 'ERROR');
            curl_close($ch);
            return false;
        } else {
            $module->writeInlogFile($module->l('Request response').$result_response, 'INFO');
            $results = json_decode($result_response, true);
            curl_close($ch);

            if ($results){
                return $results;
            } else {
                $module->writeInlogFile($module->l('Decoding JSON failed with the following message: ').json_last_error_msg(), 'ERROR');
                return false;
            }
        }

    }
}
