<?php

require_once dirname(__FILE__)."/classes/radiusdelivery_core.php";


if (!defined('_PS_VERSION_')) {
    exit;
}

class Radiusdelivery extends RadiusdeliveryCore
{
    public $id_carrier;

    protected $_successes = array();
    protected $prices = array();

    public function __construct()
    {
        $this->name = 'radiusdelivery';
        $this->tab = 'shipping_logistics';
        $this->version = '1.7.2';
        $this->author = 'MARICHAL Emmanuel';
        $this->module_key = '2a08446859385b651b3ffbc4152cb3ff';
        $this->need_instance = 0;
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Delivery by distance/time');
        $this->description = $this->l('Shipping fees based on distance or time');

        $this->ps_versions_compliancy = array('min' => '1.6', 'max' => _PS_VERSION_);

        if (!Configuration::getGlobalValue('RADIUS_DELIVERY_API_KEY')) {
            $this->warning = $this->l('Missing API Key');
        }
    }

    public function install()
    {
        if (extension_loaded('curl') == false) {
            $this->_errors[] = $this->l('You have to enable the cURL extension on your server to install this module. Please contact your webmaster.');
            return false;
        }

        return Db::getInstance()->Execute('
                CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.pSQL($this->name).'` (
                    `id_address` int(11) NOT NULL,
                    `id_carrier` int(11) NOT NULL,
                    `distance` int(11) NOT NULL DEFAULT 0,
                    `duration` int(11) NOT NULL DEFAULT 0,
                    KEY `radiuskey` (`id_address`,`id_carrier`)
                ) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=utf8 AUTO_INCREMENT=1;') &&
            parent::install() &&
            $this->registerHook('updateCarrier') &&
            $this->registerHook('actionFrontControllerAfterInit') &&
            $this->registerHook('actionAdminCarriersListingResultsModifier') &&
            $this->registerHook('actionAdminControllerSetMedia') &&
            $this->registerHook('actionObjectAddressDeleteAfter') &&
            $this->registerHook('actionObjectAddressUpdateAfter') &&
            $this->registerHook('actionObjectCarrierUpdateBefore') &&
            $this->registerHook('actionObjectCarrierUpdateAfter');
    }

    // Fix a stupid PrestaShop bug where a cart isn't invalidated when a carrier is edited/deleted.
    // Carts are still using the old carrier id and the customer can validate the order.
    // PrestaShop then selects the first valid/default carrier since the current carrier is deleted
    public function hookActionFrontControllerAfterInit($params)
    {
        $id_carrier = $this->context->cart->id_carrier;
        if ($id_carrier) {
            $query = new DbQuery();
            $query->select('checkout_session_data')->from('cart')->where('id_carrier = '.(int)$id_carrier);
            $session = json_decode(Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query), true);
            if ($session && isset($session['checkout-delivery-step']) && $session['checkout-delivery-step']['step_is_complete']) {
                $carrier = new Carrier($id_carrier);
                if ((int)$carrier->deleted) {
                    $session['checkout-delivery-step']['step_is_complete'] = false;
                    $session['checkout-payment-step']['step_is_complete'] = false;
                    $session['checkout-payment-step']['step_is_reachable'] = false;
                    Db::getInstance()->execute(
                        'UPDATE '._DB_PREFIX_.'cart SET checkout_session_data = "'.pSQL(json_encode($session)).'"
                            WHERE id_cart = '.(int)$this->context->cart->id
                    );
                }
            }
        }
    }

    // Change free shipping status in carriers list
    public function hookActionAdminCarriersListingResultsModifier($params)
    {
        $carriers = self::getCarriers();
        foreach ($params['list'] as &$carrier) {
            if (isset($carriers[(int)$carrier['id_carrier']])) {
                $carrier['is_free'] = $carriers[(int)$carrier['id_carrier']]['billing'] == self::PRICE_FREE;
            }
        }
    }

    // Override carrier wizard's free switch to avoid merchants messing with it.
    // Free shipping should be done in RadiusDelivery's module.
    public function hookActionAdminControllerSetMedia()
    {
        $id_carrier = Tools::getValue('id_carrier');
        if (get_class($this->context->controller) == 'AdminCarrierWizardController' && Tools::getValue('id_carrier')) {
            $carriers = self::getCarriers();
            if (isset($carriers[$id_carrier])) {
                $this->context->controller->addJs($this->_path.'views/js/disable_free_shipping.js');
                if ($carriers[$id_carrier]['billing'] == self::PRICE_FREE) {
                    $this->context->controller->addJs($this->_path.'views/js/check_free_shipping.js');
                }
            }
        }
    }

    // Make sure that a RadiusDelivery carrier is updated with a free status
    // causing ranges to be deleted
    public function hookActionObjectCarrierUpdateBefore($params)
    {
        $carriers = self::getCarriers();
        if (isset($carriers[$params['object']->id]) && $params['object']->is_free) {
            $params['object']->is_free = false;
        }
    }

    public function hookActionObjectCarrierUpdateAfter($params)
    {
        $carriers = self::getCarriers();
        if (Tools::getValue('deletecarrier') && $params['object']->deleted && isset($carriers[$params['object']->id])) {
            unset($carriers[$params['object']->id]);
            Configuration::updateGlobalValue('RADIUS_DELIVERY_CARRIERS', json_encode($carriers));
            Db::getInstance()->delete($this->name, 'id_carrier = '.(int)$params['object']->id);
        }
    }

    public function hookUpdateCarrier($params)
    {
        $carriers = self::getCarriers();
        if (isset($carriers[$params['id_carrier']])) {
            $carriers[$params['carrier']->id] = $carriers[$params['id_carrier']];
            unset($carriers[$params['id_carrier']]);
            Configuration::updateGlobalValue('RADIUS_DELIVERY_CARRIERS', json_encode($carriers));
            Db::getInstance()->update(
                $this->name,
                array('id_carrier' => (int)$params['carrier']->id),
                'id_carrier = '.(int)$params['id_carrier']
            );
        }
    }

    public function hookActionObjectAddressDeleteAfter($params)
    {
        Db::getInstance()->delete($this->name, 'id_address = '.(int)$params['object']->id);
    }

    public function hookActionObjectAddressUpdateAfter($params)
    {
        Db::getInstance()->delete($this->name, 'id_address = '.(int)$params['object']->id);
    }

    public function geocodeAddress($address)
    {
        $params = array(
            'q' => $address,
            'apikey' => Configuration::getGlobalValue('RADIUS_DELIVERY_API_KEY'),
        );

        try {
            $json = Tools::file_get_contents(self::GEOCODER_URL.'?'.http_build_query($params));
            $json = json_decode($json);

            if (isset($json->error)) {
                PrestaShopLogger::addLog($address.' '.json_encode($json, JSON_UNESCAPED_SLASHES), 1, null, null, null, true);
                return false;
            }
            if (!isset($json, $json->items, $json->items[0], $json->items[0]->position)) {
                PrestaShopLogger::addLog($address.' '.json_encode($json, JSON_UNESCAPED_SLASHES), 1, null, null, null, true);
                return false;
            }

            if ($json->items[0]->position->lat &&$json->items[0]->position->lng) {
                return $json->items[0]->position;
            }
        } catch (Exception $e) {
        }

        return false;
    }

    public function calculateMatrix($origin, $destination, $returnJson = false)
    {
        $params = array(
            'transportMode' => 'car',
            'origin' => $origin,
            'destination' => $destination,
            'return' => 'summary',
            'routingMode' => 'fast',
            'apikey' => Configuration::getGlobalValue('RADIUS_DELIVERY_API_KEY'),
            'units' => 'metric',
        );

        try {
            $json = Tools::file_get_contents(self::ROUTE_URL.'?'.http_build_query($params));
            $json = json_decode($json);

            if ($returnJson) {
                return $json;
            }

            if (isset($json->error)) {
                PrestaShopLogger::addLog($origin.','.$destination.' '.json_encode($json, JSON_UNESCAPED_SLASHES), 1, null, null, null, true);
                return false;
            }
            $summary = $json->routes[0]->sections[0]->summary;
            return array(
                'duration' => $summary->baseDuration,
                'distance' => $summary->length,
            );
        } catch (Exception $e) {
        }

        return false;
    }

    protected function formatAddress($id_address)
    {
        $address = new Address($id_address);

        if (!Validate::isLoadedObject($address)) {
            return false;
        }

        $state = State::getNameById($address->id_state) || '';
        return implode(', ', array(
            $address->address1,
            $address->address2,
            $address->city,
            $state . ' ' . $address->postcode,
            $address->country,
        ));
    }

    public function getOrderShippingCostExternal($cart)
    {
        return $this->getOrderShippingCost($cart, 0);
    }

    protected function addCarrier($name)
    {
        $carrier = new Carrier();

        $carrier->name = $name;
        $carrier->is_module = true;
        $carrier->active = 1;
        $carrier->need_range = true;
        $carrier->shipping_external = true;
        $carrier->range_behavior = 0;
        $carrier->external_module_name = $this->name;
        $carrier->shipping_method = 2;
        $carrier->shipping_handling = 0;

        foreach (Language::getLanguages() as $lang) {
            $carrier->delay[$lang['id_lang']] = 'Quick';
        }

        if (!$carrier->add()) {
            return false;
        }

        // Add groups
        $groups_ids = array();
        foreach (Group::getGroups($this->context->language->id) as $group) {
            $groups_ids[] = $group['id_group'];
        }

        $carrier->setGroups($groups_ids);

        // Add zones
        foreach (Zone::getZones() as $zone) {
            $carrier->addZone($zone['id_zone']);
        }

        // Add ranges
        $range_price = new RangePrice();
        $range_price->id_carrier = $carrier->id;
        $range_price->delimiter1 = '0';
        $range_price->delimiter2 = '100000';
        $range_price->add();

        $range_weight = new RangeWeight();
        $range_weight->id_carrier = $carrier->id;
        $range_weight->delimiter1 = '0';
        $range_weight->delimiter2 = '100000';
        $range_weight->add();

        return $carrier;
    }

    public function renderMainView()
    {
        $carriers = self::getCarriers();
        $weight_unit = Configuration::get('PS_WEIGHT_UNIT');

        if (!$carriers) {
            $carriers = array();
        }

        foreach ($carriers as $id_carrier => &$carrier) {
            $c = new Carrier($id_carrier);

            if (!Validate::isLoadedObject($c)) {
                unset($carriers[$id_carrier]);
                continue;
            }

            $carrier['id_carrier'] = $c->id;
            $carrier['carrier_name'] = $c->name;

            $price_mode = '';
            switch ($carrier['price_mode']) {
                case self::PRICE_MODE_ROUNDED:
                    $price_mode .= ' ('.$this->l('Rounded').')';
                    break;
                case self::PRICE_MODE_PSYCHOLOGICAL:
                    $price_mode .= ' ('.$this->l('Psychological').')';
                    break;
            }

            switch ($carrier['billing']) {
                case self::PRICE_FREE:
                    $carrier['billing'] = $this->l('Free');
                    break;

                case self::PRICE_NATIVE:
                    $carrier['billing'] = $this->l('Native (weight/price)');
                    break;

                case self::PRICE_DISTANCE:
                    $carrier['billing'] = $carrier['base_price'] ? Tools::displayPrice($carrier['base_price']).' + ' : '';
                    $carrier['billing'] .= Tools::displayPrice($carrier['price']).' / km'.$price_mode;
                    break;

                case self::PRICE_DURATION:
                    $carrier['billing'] = $carrier['base_price'] ? Tools::displayPrice($carrier['base_price']).' + ' : '';
                    $carrier['billing'] .= Tools::displayPrice($carrier['price']).' / '.$this->l('minute').$price_mode;
                    break;

                case self::PRICE_DURATION_RANGE:
                    $carrier['billing'] = $this->l('Ranges (time)');
                    if (!empty($carrier['ranges'])) {
                        $carrier['min_duration'] = $carrier['ranges'][0]['min'];
                        $carrier['max_duration'] = $carrier['ranges'][count($carrier['ranges']) - 1]['max'];
                    }
                    break;

                case self::PRICE_DISTANCE_RANGE:
                    $carrier['billing'] = $this->l('Ranges (distance)');
                    if (!empty($carrier['ranges'])) {
                        $carrier['min_distance'] = $carrier['ranges'][0]['min'];
                        $carrier['max_distance'] = $carrier['ranges'][count($carrier['ranges']) - 1]['max'];
                    }
                    break;
            }

            $carrier['limit_min'] = $carrier['limit_max'] = [];
            if (isset($carrier['min_price']) && $carrier['min_price']) {
                $carrier['limit_min'][] = Tools::displayPrice($carrier['min_price']);
            }
            if (isset($carrier['min_weight']) && $carrier['min_weight']) {
                $carrier['limit_min'][] = $carrier['min_weight'].' '.$weight_unit;
            }
            if (isset($carrier['max_price']) && $carrier['max_price']) {
                $carrier['limit_max'][] = Tools::displayPrice($carrier['max_price']);
            }
            if ($c->max_weight > 0) {
                $carrier['limit_max'][] = (float)$c->max_weight.' '.$weight_unit;
            }
            if (isset($carrier['min_distance']) && $carrier['min_distance']) {
                $carrier['limit_min'][] = $carrier['min_distance'].' km';
            }
            if (isset($carrier['min_duration']) && $carrier['min_duration']) {
                $carrier['limit_min'][] = $carrier['min_duration'].' min';
            }
            if ($carrier['max_distance']) {
                $carrier['limit_max'][] = $carrier['max_distance'].' km';
            }
            if ($carrier['max_duration']) {
                $carrier['limit_max'][] = $carrier['max_duration'].' min';
            }

            $carrier['limit_min'] = count($carrier['limit_min']) ? implode(' / ', $carrier['limit_min']) : '-';
            $carrier['limit_max'] = count($carrier['limit_max']) ? implode(' / ', $carrier['limit_max']) : '-';
        }

        $this->smarty->assign(array(
            'current_url' => $this->context->link->getAdminLink('AdminModules').'&configure='.$this->name.'&tab_module=shipping_logistics&module_name='.$this->name,
            'banner_url' => '../modules/radiusdelivery/views/img/banner_'.($this->context->language->iso_code === 'fr' ? 'fr' : 'en').'.png',
            'carriers' => json_encode(array(
                'columns' => array(
                    array('content' => 'ID', 'key' => 'id_carrier', 'center' => true),
                    array('content' => $this->l('Carrier'), 'key' => 'carrier_name'),
                    array('content' => $this->l('Billing'), 'key' => 'billing'),
                    array('content' => $this->l('Limit min'), 'key' => 'limit_min'),
                    array('content' => $this->l('Limit max'), 'key' => 'limit_max'),
                ),
                'rows' => $carriers,
                'rows_actions' => array(
                    array('title' => $this->l('Edit'), 'action' => 'edit_carrier', 'icon' => 'pencil'),
                    array('title' => $this->l('Delete'), 'action' => 'delete_carrier', 'icon' => 'trash')
                ),
                'top_actions' => array(
                    array('title' => $this->l('Add carrier'), 'action' => 'add_carrier', 'icon' => 'add'),
                ),
                'url_params' => array('configure' => $this->name),
                'identifier' => 'id_carrier'
            ))
        ));

        $debug_address_form = '';
        if (count($carriers)) {
            $debug_address_form = $this->renderDebugAddressForm();
        }

        $hasPaymentsWithRestrictions = '';
        if ($this->hasPaymentsWithRestrictions($carriers)) {
            $warning_msg = $this->l('Some of your payment methods are not visible with your carrier(s) because of restrictions set up on your shop. To change this, go in Payment > Preferences.');
            if (method_exists($this, 'displayWarning')) {
                $hasPaymentsWithRestrictions = $this->displayWarning($warning_msg);
            } else {
                $hasPaymentsWithRestrictions = $this->displayError($warning_msg);
            }
        }

        return $hasPaymentsWithRestrictions.$this->display(__FILE__, 'views/templates/admin/configure.tpl').$debug_address_form.$this->renderSettingsForm();
    }

    public function renderAddCarrier($id_carrier)
    {
        $this->context->controller->addJS($this->_path.'views/js/add_carrier.js');

        $rule = $carrier = false;
        if ($id_carrier) {
            $carrier = new Carrier($id_carrier);
            if (!Validate::isLoadedObject($carrier)) {
                $carrier = false;
            } else {
                $carriers = self::getCarriers();
                if (isset($carriers[$id_carrier])) {
                    $rule = $carriers[$id_carrier];
                } else {
                    $carrier = false;
                }
            }
        }

        $this->smarty->assign(array(
            'current_url' => $this->context->link->getAdminLink('AdminModules').'&configure='.$this->name.'&tab_module=shipping_logistics&module_name='.$this->name,
            'currency' => $this->context->currency,
            'units' => array(
                'weigth' => Configuration::get('PS_WEIGHT_UNIT'),
            ),
            'carrier' => $carrier,
            'rule' => $rule,
        ));

        return $this->display(__FILE__, 'views/templates/admin/add_carrier.tpl');
    }

    protected function isRange($billing)
    {
        return in_array($billing, array('PRICE_DURATION_RANGE', 'PRICE_DISTANCE_RANGE'));
    }

    public function saveCarrier($id_carrier = false)
    {
        $longitude = Tools::ps_round((float)Tools::getValue('longitude'), 8);
        $latitude = Tools::ps_round((float)Tools::getValue('latitude'), 8);
        $min_billing_distance = (float)Tools::getValue('min_billing_distance');
        $min_billing_duration = (float)Tools::getValue('min_billing_duration');
        $min_billing_deduct = (bool)Tools::getValue('min_billing_deduct');
        $min_price = (float)Tools::getValue('min_price');
        $max_price = (float)Tools::getValue('max_price');
        $min_weight = (float)Tools::getValue('min_weight');
        $max_weight = (float)Tools::getValue('max_weight');

        if (!Validate::isCoordinate($longitude) || !Validate::isCoordinate($latitude)) {
            $this->_errors[] = $this->l('The coordinates are incorrect.');
            return;
        }

        $billing = Tools::getValue('billing');
        if ($this->isRange($billing)) {
            $range_price = Tools::getValue('range_price');
            $range_min = Tools::getValue('range_min');
            $range_max = Tools::getValue('range_max');
            $price = $min_distance = $min_duration = $max_distance = $max_duration = $base_price = 0;
            $price_mode = false;

            if (count($range_price) !== count($range_min) || count($range_price) !== count($range_max)) {
                $this->_errors[] = $this->l('You must fill all the ranges correctly');
                return;
            }

            foreach (array_merge($range_price, array_merge($range_min, $range_max)) as $range) {
                if (!Validate::isFloat($range) && !Validate::isInt($range)) {
                    $this->_errors[] = $this->l('All values in ranges must be valid numbers');
                    return;
                }
            }
        } else {
            $min_distance = $min_duration = $max_distance = $max_duration = 0;
            $price = (float)Tools::getValue('price');
            $base_price = (float)Tools::getValue('base_price');
            if ($billing == self::PRICE_DISTANCE || $billing == self::PRICE_FREE) {
                $min_distance = (float)Tools::getValue('min_distance');
                $max_distance = (float)Tools::getValue('max_distance');
            }
            if ($billing == self::PRICE_DURATION || $billing == self::PRICE_FREE) {
                $min_duration = (int)Tools::getValue('min_duration');
                $max_duration = (int)Tools::getValue('max_duration');
            }
            $price_mode = Tools::getValue('price_mode');
        }

        $carrier = new Carrier($id_carrier);
        if (!Validate::isLoadedObject($carrier)) {
            $name = Tools::getValue('name');
            if (!$name || !Validate::isCarrierName($name)) {
                $this->_errors[] = $this->l('Name is invalid');
                return;
            }

            $carrier = $this->addCarrier($name);
            if (!$carrier) {
                $this->_errors[] = $this->l('Can\'t create carrier');
                return;
            }
            $id_carrier = $carrier->id;
        } else {
            $carrier->max_weight = $max_weight;
            $carrier->save();
        }

        $carriers = self::getCarriers();

        if (!isset($carriers[$id_carrier])) {
            $carriers[$id_carrier] = array();
        }

        if (empty($carriers[$id_carrier]) || $latitude != $carriers[$id_carrier]['latitude'] || $longitude != $carriers[$id_carrier]['longitude']) {
            $carriers[$id_carrier]['latitude'] = $latitude;
            $carriers[$id_carrier]['longitude'] = $longitude;
            Db::getInstance()->delete($this->name, 'id_carrier = '.(int)$id_carrier);
        } elseif (Validate::isLoadedObject($carrier) && $id_carrier != $carrier->id) {
            Db::getInstance()->update($this->name, array(
                'id_carrier' => (int)$id_carrier,
            ), 'id_carrier = '.(int)$carrier->id);
        }

        $carriers[$id_carrier]['billing'] = $billing;
        $carriers[$id_carrier]['price'] = $price;
        $carriers[$id_carrier]['base_price'] = $base_price;
        $carriers[$id_carrier]['min_billing_distance'] = $min_billing_distance;
        $carriers[$id_carrier]['min_billing_duration'] = $min_billing_duration;
        $carriers[$id_carrier]['min_billing_deduct'] = $min_billing_deduct;
        $carriers[$id_carrier]['min_distance'] = $min_distance;
        $carriers[$id_carrier]['min_duration'] = $min_duration;
        $carriers[$id_carrier]['max_distance'] = $max_distance;
        $carriers[$id_carrier]['max_duration'] = $max_duration;
        $carriers[$id_carrier]['min_weight'] = $min_weight;
        $carriers[$id_carrier]['min_price'] = $min_price;
        $carriers[$id_carrier]['max_price'] = $max_price;
        $carriers[$id_carrier]['price_mode'] = $price_mode;

        if ($this->isRange($billing)) {
            $carriers[$id_carrier]['ranges'] = array();
            $range_price_length = count($range_price);
            for ($i = 0; $i < $range_price_length; $i++) {
                $carriers[$id_carrier]['ranges'][] = array(
                    'min' => $range_min[$i],
                    'max' => $range_max[$i],
                    'price' => $range_price[$i]
                );
            }
        }

        Configuration::updateGlobalValue('RADIUS_DELIVERY_CARRIERS', json_encode($carriers));
        $this->_successes[] = $this->l('Carrier saved with success');
    }

    protected function deleteCarrier($id_carrier)
    {
        $carriers = self::getCarriers();
        if ($carriers && isset($carriers[$id_carrier])) {
            $c = new Carrier($id_carrier);
            if (Validate::isLoadedObject($c)) {
                $c->deleted = true;
                $c->save();
                Warehouse::removeCarrier($c->id_reference);
            }
            unset($carriers[$id_carrier]);
            Configuration::updateGlobalValue('RADIUS_DELIVERY_CARRIERS', json_encode($carriers));
            Db::getInstance()->delete($this->name, 'id_carrier = '.(int)$id_carrier);
            return true;
        }
        return false;
    }

    protected function hasPaymentsWithRestrictions($carriers)
    {
        if (version_compare(_PS_VERSION_, 1.7, '<')) {
            return false;
        }

        $payment_ids = array_map(function ($module) {
            return $module['id_module'];
        }, Module::getPaymentModules());
        $carrier_ids = array_keys($carriers);

        if (empty($payment_ids) || empty($carrier_ids)) {
            return false;
        }

        $query = new DbQuery();
        $query->select('*');
        $query->from('module_carrier');
        $query->where('id_reference IN ('.implode(',', array_map('intval', $carrier_ids)).') AND id_module IN ('.implode(',', array_map('intval', $payment_ids)).')');

        return count($payment_ids) != count(Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query));
    }

    protected function debugAddress($id_address, $id_carrier)
    {
        $carriers = self::getCarriers();

        if (!$carriers || !isset($carriers[$id_carrier])) {
            $this->_errors[] = $this->l('Invalid carrier');
            return '';
        }

        $address = $this->formatAddress($id_address);

        if (!$address) {
            $this->_errors[] = $this->l('This address doesn\'t exist');
            return '';
        }

        $geocodeAddress = $this->geocodeAddress($address);
        if (!$geocodeAddress) {
            $this->_errors[] = $this->l('This address doesn\'t seem to be valid as the HERE API isn\'t able to recognize it.');
            return '';
        }

        $origin = $carriers[$id_carrier]['latitude'].','.$carriers[$id_carrier]['longitude'];
        $matrix = $this->calculateMatrix($origin, $geocodeAddress->lat.','.$geocodeAddress->lng, true);

        return json_encode($matrix, JSON_PRETTY_PRINT);
    }

    protected function renderDebugAddressForm()
    {
        $carrier_ids = array_keys(self::getCarriers());

        $query = new DbQuery();
        $query->select('id_carrier, name');
        $query->from('carrier');
        $query->where('id_carrier IN ('.implode(',', array_map('intval', $carrier_ids)).')');

        $carriers = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        $settings = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Debug'),
                    'icon' => 'icon-info'
                ),
                'input' => array(
                    array(
                        'type' => 'html',
                        'name' => 'desc',
                        'html_content' => $this->l('Use this section to get the result of the API between your carrier and an adress. It can be a little bit technical but it will give you all the informations (time, distance, coordinates, ect...) to see what\'s the issue for a specific address. It will give you the details of the error if the HERE API isn\'t able to process it. Durations are in seconds and distances are in meters.'),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Carrier'),
                        'name' => 'id_carrier',
                        'options' => array(
                            'query' => $carriers,
                            'id' => 'id_carrier',
                            'name' => 'name',
                        ),
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Address ID'),
                        'name' => 'id_address',
                        'class' => 'fixed-width-sm',
                    ),
                ),
                'submit' => array(
                    'name' => 'debugAddress',
                    'title' => $this->l('Debug')
                )
            ),
        );

        $helper = new HelperForm();

        $helper->id = 'debugForm';
        $helper->show_toolbar = false;
        $helper->table = 'settings';
        $lang = new Language((int)Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false).'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->submit_action = 'submitDebugAddress';

        $helper->tpl_vars = array(
            'fields_value' => array(
                'id_address' => '',
                'id_carrier' => '',
            ),
        );

        if (Tools::isSubmit('debugAddress')) {
            $settings['form']['input'][] = array(
                'type' => 'html',
                'name' => 'result',
                'html_content' => nl2br(str_replace(' ', '&nbsp;', $this->debugAddress(Tools::getValue('id_address'), Tools::getValue('id_carrier')))),
            );
        }

        return $helper->generateForm(array($settings));
    }

    protected function renderSettingsForm()
    {
        $conf = Configuration::getMultiple(array(
            'RADIUS_DELIVERY_TAXES_EXCLUDED',
            'RADIUS_DELIVERY_API_KEY',
            'RADIUS_DELIVERY_DISCOUNTS_DEDUCTED',
        ));

        $create_api_url = str_replace('ISO_CODE', $this->context->language->iso_code, self::CREATE_API_KEY_URL);
        $iso_lang = Language::getIsoById($this->context->language->id);
        $doc_iso = file_exists(_PS_MODULE_DIR_.$this->name.'/docs/readme_'.$iso_lang.'.pdf') ? $iso_lang : 'en';


        $settings = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Settings'),
                    'icon' => 'icon-cog'
                ),
                'input' => array(
                    array(
                        'type' => 'textbutton',
                        'label' => $this->l('API key'),
                        'name' => 'api_key',
                        'desc' => $this->l('You can create an API key (REST but do not enable the "trusted domain" setting) by clicking the "Create API key" button. Start with the free plan, it allows you to check up to 125000 addresses per month (1 address = 2 API transactions). The module is optimized to check only once every address to avoid to exceed quotas.'),
                        'button' => array(
                            'label' => $this->l('Create API key'),
                            'attributes' => array(
                                'onclick' => 'window.open("'.$create_api_url.'", "_blank");',
                            )
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Taxes excluded'),
                        'name' => 'taxes_excluded',
                        'desc' => $this->l('If you want the order amount limitation to be calculated without taxes.'),
                        'values' => array(
                            array(
                                'id' => 'taxes_excluded_on',
                                'value' => true,
                            ),
                            array(
                                'id' => 'taxes_excluded_off',
                                'value' => false,
                            ),
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Discounts deducted'),
                        'name' => 'discounts_deducted',
                        'desc' => $this->l('If you want the order amount limitation to be calculated on the total with discounts deducted'),
                        'values' => array(
                            array(
                                'id' => 'discounts_deducted_on',
                                'value' => true,
                            ),
                            array(
                                'id' => 'discounts_deducted_off',
                                'value' => false,
                            ),
                        ),
                    ),
                ),
                'buttons' => array(
                    'documentation' => array(
                        'href' => '../modules/'.$this->name.'/docs/readme_'.$doc_iso.'.pdf',
                        'target' => '_blank',
                        'title' => $this->l('View documentation')
                    )
                ),
                'submit' => array(
                    'name' => 'saveSettings',
                    'title' => $this->l('Save')
                )
            ),
        );

        $helper = new HelperForm();

        $helper->id = 'settingsForm';
        $helper->show_toolbar = false;
        $helper->table = 'settings';
        $lang = new Language((int)Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false).'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->submit_action = 'submitSaveSettings';

        $helper->tpl_vars = array(
            'fields_value' => array(
                'api_key' => $conf['RADIUS_DELIVERY_API_KEY'],
                'taxes_excluded' => $conf['RADIUS_DELIVERY_TAXES_EXCLUDED'],
                'discounts_deducted' => $conf['RADIUS_DELIVERY_DISCOUNTS_DEDUCTED'],
            ),
        );

        return $helper->generateForm(array($settings));
    }

    public function getContent()
    {
        $this->context->controller->addJS($this->_path.'views/js/riot.js');
        $this->context->controller->addCSS($this->_path.'views/css/back.css');
        $action = Tools::getValue('action');
        $html = '';

        if (Tools::isSubmit('saveSettings')) {
            $apikey = Tools::getValue('api_key');
            if (!$apikey) {
                $this->_errors[] = $this->l('API Key is not valid');
            }
            if (!count($this->_errors)) {
                if ($apikey !== Configuration::getGlobalValue('RADIUS_DELIVERY_API_KEY')) {
                    Db::getInstance()->execute('TRUNCATE TABLE '._DB_PREFIX_.pSQL($this->name));
                }
                Configuration::updateGlobalValue('RADIUS_DELIVERY_API_KEY', $apikey);
                Configuration::updateGlobalValue('RADIUS_DELIVERY_TAXES_EXCLUDED', Tools::getValue('taxes_excluded'));
                Configuration::updateGlobalValue('RADIUS_DELIVERY_DISCOUNTS_DEDUCTED', Tools::getValue('discounts_deducted'));
                $html .= $this->displayConfirmation($this->l('Saved with success'));
            }
        }

        if ($action === 'delete_carrier' && Tools::getValue('id_carrier')) {
            $this->deleteCarrier(Tools::getValue('id_carrier'));
        }

        if (Tools::isSubmit('submitCarrier')) {
            $this->saveCarrier(Tools::getValue('id_carrier'));
        }

        $conf = Configuration::getMultiple(array(
            'RADIUS_DELIVERY_TAXES_EXCLUDED',
            'RADIUS_DELIVERY_API_KEY',
            'RADIUS_DELIVERY_DISCOUNTS_DEDUCTED',
        ));

        if ($conf['RADIUS_DELIVERY_API_KEY'] && !$this->geocodeAddress('Boulevard de Parc, 77700 Coupvray, France')) {
            $this->_errors[] = $this->l('API Key is not valid');
        }

        $iso_lang = Language::getIsoById($this->context->language->id);
        $doc_iso = file_exists(_PS_MODULE_DIR_.$this->name.'/docs/readme_'.$iso_lang.'.pdf') ? $iso_lang : 'en';
        $this->smarty->assign(array(
            'doc_link' => '../modules/'.$this->name.'/docs/readme_'.$doc_iso.'.pdf',
            'conf' => $conf,
            'CREATE_API_KEY_URL' => str_replace('ISO_CODE', $this->context->language->iso_code, self::CREATE_API_KEY_URL)
        ));

        if (in_array($action, array('add_carrier', 'edit_carrier'))) {
            $html .= $this->renderAddCarrier(Tools::getValue('id_carrier'));
        } else {
            $html .= $this->renderMainView();
        }

        $successes = implode('', array_map(array($this, 'displayConfirmation'), $this->_successes));

        $errors = implode('', array_map(array($this, 'displayError'), $this->_errors));

        return $successes.$errors.$html.$this->display(__FILE__, 'views/templates/admin/prestui/ps-tags.tpl');
    }
}
